<?php

namespace App\Http\Controllers\Admin;

use Pusher\Pusher;
use App\Models\Hiring;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Models\GeneralSetting;
use App\Models\HiringConversation;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class ManageHiringController extends Controller
{
    public function index()
    {
        $this->pageTitle = 'All Hiring';
        return $this->filterHiring();
    }

    public function pending()
    {
        $this->pageTitle = 'Pending Hiring';
        return $this->filterHiring('pending');
    }

    public function inprogress()
    {
        $this->pageTitle = 'Inprogress Hiring';
        return $this->filterHiring('inprogress');
    }

    public function jobDone()
    {
        $this->pageTitle = 'Job Done Hiring';
        return $this->filterHiring('JobDone');
    }

    public function completed()
    {
        $this->pageTitle = 'Completed Hiring';
        return $this->filterHiring('completed');
    }

    public function reported()
    {
        $this->pageTitle = 'Reported Hiring';
        return $this->filterHiring('reported');
    }

    public function cancelled()
    {
        $this->pageTitle = 'Cancelled Hiring';
        return $this->filterHiring('cancelled');
    }

    public function detail($id)
    {
        $pageTitle  = 'Hiring Detail';
        $hiring   = Hiring::with('user', 'influencer')->findOrFail($id);
        $conversations = HiringConversation::where('hiring_id', $hiring->id)->orderBy('id', 'desc')->get();
        return view('admin.hiring.detail', compact('pageTitle', 'hiring','conversations'));
    }

    protected function filterHiring($scope = null)
    {
        $hirings = Hiring::paymentCompleted();

        if ($scope) {
            $hirings = $hirings->$scope();
        }
        $hirings = $hirings->with('user', 'influencer')->latest()->paginate(getPaginate());
        $pageTitle = $this->pageTitle;

        return view('admin.hiring.index', compact('pageTitle', 'hirings'));
    }

    public function takeAction($id, $status)
    {
        $hiring = Hiring::with('user', 'influencer')->findOrFail($id);

        if ($status == 1) {
            $this->inFavourOfInfluencer($hiring);
        }

        if ($status == 6) {
            $this->inFavourOfClient($hiring);
        }

        $hiring->status = $status;
        $hiring->save();

        $notify[] = ['success', 'Action taken successfully'];
        return back()->withNotify($notify);
    }

    protected function inFavourOfClient($hiring)
    {
        $influencer = $hiring->influencer;
        $user       = $hiring->user;
        $user->balance += $hiring->amount;
        $user->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $hiring->amount;
        $transaction->post_balance = $user->balance;
        $transaction->trx_type     = '+';
        $transaction->details      = 'Payment refunded due to incomplete hiring task';
        $transaction->trx          = getTrx();
        $transaction->remark       = 'payment_refunded';
        $transaction->save();

        $general = gs();

        notify($user, 'HIRING_REFUND', [
            'site_currency' => $general->cur_text,
            'title'         => $hiring->title,
            'amount'        => showAmount($hiring->amount),
            'post_balance'  => showAmount($user->balance),
            'hiring_no'     => $hiring->hiring_no,
        ]);

        notify($influencer, 'HIRING_REJECTED', [
            'site_currency' => $general->cur_text,
            'amount'        => showAmount($hiring->amount),
            'post_balance'  => showAmount($influencer->balance),
            'hiring_no'     => $hiring->hiring_no,
            'title'         => $hiring->title,
        ]);
    }

    protected function inFavourOfInfluencer($hiring)
    {
        $influencer = $hiring->influencer;
        $user       = $hiring->user;

        $influencer->balance += $hiring->amount;
        $influencer->increment('completed_order');
        $influencer->save();

        $transaction                = new Transaction();
        $transaction->influencer_id = $influencer->id;
        $transaction->amount        = $hiring->amount;
        $transaction->post_balance  = $influencer->balance;
        $transaction->trx_type      = '+';
        $transaction->details       = 'Payment received for completing hiring task';
        $transaction->trx           = getTrx();
        $transaction->remark        = 'payment_on_hiring';
        $transaction->save();

        $general = gs();

        $shortCodes = [
            'site_currency' => $general->cur_text,
            'amount'        => showAmount($hiring->amount),
            'hiring_no'     => $hiring->hiring_no,
            'title'         => $hiring->title,
        ];

        notify($influencer, 'HIRING_COMPLETED_INFLUENCER', $shortCodes);
        notify($user, 'HIRING_COMPLETED_CLIENT', $shortCodes);
    }

    public function conversationStore(Request $request)
    {
        $hiring    = Hiring::find($request->hiring_id);
        $admin = auth()->guard('admin')->user();

        $validator = Validator::make($request->all(), [
            'message' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()]);
        }

        $message                = new HiringConversation();
        $message->hiring_id     = $hiring->id;
        $message->user_id       = $hiring->user_id;
        $message->influencer_id = $hiring->influencer_id;
        $message->admin_id      = auth()->guard('admin')->id();
        $message->sender        = 'admin';
        $message->message       = $request->message;
        $message->save();

        $senderImagePath = getImage(getFilePath('adminProfile') . '/' . ($admin ->image ?? ''), getFileSize('adminProfile'));
        $sendTime = date('h:i a',strtotime($message->created_at));
        $general = GeneralSetting::first();


        $orderData = [
            'user_id' => $hiring->user_id,
            'influencer_id' => $hiring->influencer_id
        ];



        foreach($orderData as $item)
        {
            $options = [
                'cluster' => $general->pusher_credential->app_cluster,
                'encrypted' => $general->pusher_credential->useTLS
            ];

            $pusher = new Pusher(
                $general->pusher_credential->app_key,
                $general->pusher_credential->app_secret,
                $general->pusher_credential->app_id,
                $options
            );

            $data = [
                'id' => $message->id,
                'message' => $message->message,
                'sender' => $message->admin_id,
                'senderName' => "admin",
                'receiver' =>  $item,
                'created_at' => $message->created_at,
                'senderImag' => $senderImagePath,
                'sendTime' => $sendTime,
            ];


            $event_name = '' . $item . '';
            $pusher->trigger($event_name, "App\\Events\\HiringChat", $data);
        }

        return response()->json($data);
    }

}

