<?php

namespace App\Http\Controllers\Admin;

use Pusher\Pusher;
use App\Models\Order;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Models\GeneralSetting;
use App\Models\OrderConversation;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class ManageOrderController extends Controller
{
    public function index()
    {
        $this->pageTitle = 'All Orders';
        return $this->filterOrder();
    }

    public function pending()
    {
        $this->pageTitle = 'Pending Orders';
        return $this->filterOrder('pending');
    }

    public function inprogress()
    {
        $this->pageTitle = 'Inprogress Orders';
        return $this->filterOrder('inprogress');
    }

    public function jobDone()
    {
        $this->pageTitle = 'Job Done Orders';
        return $this->filterOrder('JobDone');
    }

    public function completed()
    {
        $this->pageTitle = 'Completed Orders';
        return $this->filterOrder('completed');
    }

    public function reported()
    {
        $this->pageTitle = 'Reported Orders';
        return $this->filterOrder('reported');
    }

    public function cancelled()
    {
        $this->pageTitle = 'Cancelled Orders';
        return $this->filterOrder('cancelled');
    }

    protected function filterOrder($scope = null)
    {
        $orders = Order::paymentCompleted();

        if ($scope) {
            $orders = $orders->$scope();
        }

        $orders = $orders->with('user', 'influencer')->latest()->paginate(getPaginate());

        $pageTitle = $this->pageTitle;

        return view('admin.orders.index', compact('pageTitle', 'orders'));
    }

    public function detail($id)
    {
        $pageTitle     = 'Order Detail';
        $order         = Order::with('user', 'influencer')->findOrFail($id);
        $conversations = OrderConversation::where('order_id', $order->id)->orderBy('id', 'desc')->get();
        return view('admin.orders.detail', compact('pageTitle', 'order','conversations'));
    }

    public function conversationStore(Request $request)
    {

        $order     = Order::find($request->order_id);
        $admin = auth()->guard('admin')->user();

        $validator = Validator::make($request->all(), [
            'message' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()]);
        }

        $message                = new OrderConversation();
        $message->order_id      = $order->id;
        $message->user_id       = $order->user_id;
        $message->influencer_id = $order->influencer_id;
        $message->admin_id      = auth()->guard('admin')->id();
        $message->sender        = 'admin';
        $message->message       = $request->message;
        $message->save();

        $senderImagePath = getImage(getFilePath('adminProfile') . '/' . ($admin ->image ?? ''), getFileSize('adminProfile'));
        $sendTime = date('h:i a',strtotime($message->created_at));
        $general = GeneralSetting::first();


        $orderData = [
            'user_id' => $order->user_id,
            'influencer_id' => $order->influencer_id
        ];



        foreach($orderData as $item)
        {
            $options = [
                'cluster' => $general->pusher_credential->app_cluster,
                'encrypted' => $general->pusher_credential->useTLS
            ];

            $pusher = new Pusher(
                $general->pusher_credential->app_key,
                $general->pusher_credential->app_secret,
                $general->pusher_credential->app_id,
                $options
            );

            $data = [
                'id' => $message->id,
                'message' => $message->message,
                'sender' => $message->admin_id,
                'senderName' => 'admin',
                'receiver' =>  $item,
                'created_at' => $message->created_at,
                'senderImag' => $senderImagePath,
                'sendTime' => $sendTime,
            ];


            $event_name = '' . $item . '';
            $pusher->trigger($event_name, "App\\Events\\HiringChat", $data);
        }

        return response()->json($data);
    }

    public function takeAction($id, $status)
    {

        $order = Order::with('user', 'influencer', 'service')->findOrFail($id);

        if ($status == 1) {
            $this->inFavourOfInfluencer($order);
        }

        if ($status == 6) {
            $this->inFavourOfClient($order);
        }

        $order->status = $status;
        $order->save();

        $notify[] = ['success', 'Action taken successfully'];
        return back()->withNotify($notify);
    }

    protected function inFavourOfClient($order)
    {
        $influencer = $order->influencer;
        $user       = $order->user;

        $user->balance += $order->amount;
        $user->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $order->amount;
        $transaction->post_balance = $user->balance;
        $transaction->trx_type     = '+';
        $transaction->details      = 'Payment refunded due to incomplete service';
        $transaction->trx          = getTrx();
        $transaction->remark       = 'payment_refunded';
        $transaction->save();

        notify($user, 'ORDER_REFUND', [
            'title'         => @$order->title,
            'amount'        => showAmount($order->amount),
            'post_balance'  => showAmount($user->balance),
            'order_no'      => $order->order_no,
        ]);

        notify($influencer, 'ORDER_REJECTED', [
            'title'         => @$order->title,
            'amount'        => showAmount($order->amount),
            'post_balance'  => showAmount($influencer->balance),
            'order_no'      => $order->order_no,
        ]);
    }

    protected function inFavourOfInfluencer($order)
    {

        $influencer = $order->influencer;
        $user       = $order->user;

        $influencer->balance += $order->amount;
        $influencer->increment('completed_order');
        $influencer->save();

        $transaction                = new Transaction();
        $transaction->influencer_id = $influencer->id;
        $transaction->amount        = $order->amount;
        $transaction->post_balance  = $influencer->balance;
        $transaction->trx_type      = '+';
        $transaction->details       = 'Payment received for completing service order';
        $transaction->trx           = getTrx();
        $transaction->remark        = 'order_payment';
        $transaction->save();

        $shortCodes = [
            'title'         => $order->title,
            'amount'        => showAmount($order->amount),
            'order_no'      => $order->order_no,
        ];

        notify($influencer, 'ORDER_COMPLETED_INFLUENCER', $shortCodes);
        notify($user, 'ORDER_COMPLETED_CLIENT', $shortCodes);
    }
}
