<?php

namespace App\Http\Controllers\Influencer;

use Pusher\Pusher;
use App\Models\User;
use App\Models\Order;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Models\GeneralSetting;
use App\Models\OrderConversation;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class OrderController extends Controller
{
    public function index() {
        $this->pageTitle = 'All Orders';
        return $this->filterOrder();
    }

    public function pending() {
        $this->pageTitle = 'Pending Orders';
        return $this->filterOrder('pending');
    }

    public function inprogress() {
        $this->pageTitle = 'Processing Orders';
        return $this->filterOrder('inprogress');
    }

    public function jobDone() {
        $this->pageTitle = 'Job Done Orders';
        return $this->filterOrder('JobDone');
    }

    public function completed() {
        $this->pageTitle = 'Completed Orders';
        return $this->filterOrder('completed');
    }

    public function reported() {
        $this->pageTitle = 'Reported Orders';
        return $this->filterOrder('reported');
    }

    public function cancelled() {
        $this->pageTitle = 'Cancelled Orders';
        return $this->filterOrder('cancelled');
    }

    protected function filterOrder($scope = null) {
        $influencerId = authInfluencerId();
        $orders       = Order::query();

        if ($scope) {
            $orders = $orders->$scope();
        }

        $orders = $orders->where('influencer_id', $influencerId)->with('user')->latest()->paginate(getPaginate());
        $pageTitle = $this->pageTitle;
        return view($this->activeTemplate . 'influencer.orders.index', compact('pageTitle', 'orders'));
    }

    public function detail($id) {
        $pageTitle = 'Order Detail';
        $order     = Order::where('influencer_id', authInfluencerId())->with('user', 'service')->findOrFail($id);
        return view($this->activeTemplate . 'influencer.orders.detail', compact('pageTitle', 'order'));
    }

    public function cancelOrder($id) {

        $influencer    = authInfluencer();
        $order         = Order::where('id', $id)->where('influencer_id', $influencer->id)->with('user')->firstOrFail();
        $order->status = 5;
        $order->save();

        $user    = $order->user;
        $user->balance += $order->amount;
        $user->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $order->amount;
        $transaction->post_balance = $user->balance;
        $transaction->trx_type     = '+';
        $transaction->details      = 'Payment refunded for order cancellation';
        $transaction->trx          = getTrx();
        $transaction->remark       = 'order_payment';
        $transaction->save();

        notify($user, 'ORDER_CANCELLED', [
            'influencer'    => $influencer->username,
            'amount'        => showAmount($order->amount),
            'order_no'      => $order->order_no,
            'post_balance'  => showAmount($user->balance),
            'title'         => $order->title,
        ]);

        $notify[] = ['success', 'Order canceled successfully'];
        return back()->withNotify($notify);
    }

    public function orderAccept($id) {
        $influencer    = authInfluencer();
        $order         = Order::pending()->where('id', $id)->where('influencer_id', $influencer->id)->with('user', 'service')->first();

        if(!$order){

            $notify[] = ['warning', 'Before admin payment accepted and then your action'];
            return back()->withNotify($notify);
        }

        $order->status = 2;
        $order->save();

        $user   = $order->user;

        notify($user, 'ORDER_ACCEPT', [
            'influencer'    => $influencer->username,
            'title'         => $order->title,
            'amount'        => showAmount($order->amount),
            'order_no'      => $order->order_no,
        ]);

        $notify[] = ['success', 'Order accepted successfully'];
        return back()->withNotify($notify);
    }

    public function jobDoneStatus($id) {
        $influencer    = authInfluencer();
        $order         = Order::inprogress()->where('id', $id)->where('influencer_id', $influencer->id)->with('user')->firstOrFail();
        $order->status = 3;
        $order->save();

        $user    = $order->user;

        notify($user, 'JOB_DONE', [
            'influencer'    => $influencer->username,
            'title'         => $order->title,
            'amount'        => showAmount($order->amount),
            'order_no'      => $order->order_no,
        ]);
        $notify[] = ['success', 'Order has been done successfully'];
        return back()->withNotify($notify);
    }

    public function conversation($id) {
        $pageTitle           = 'Order Conversation';
        $order               = Order::where('influencer_id', authInfluencerId())->with('orderMessage')->findOrFail($id);
        $user                = User::where('id', $order->user_id)->first();
        $conversationMessage = OrderConversation::with('influencer', 'admin', 'user')->where('order_id', $order->id)->latest()->get();
        return view($this->activeTemplate . 'influencer.orders.conversation', compact('pageTitle', 'conversationMessage', 'user', 'order'));
    }

    public function conversationStore(Request $request) {

        $order = Order::where('influencer_id', authInfluencerId())->find($request->order_id);
        $influencer = authInfluencer();

        if (!$order) {
            return response()->json(['error' => 'Order not found.']);
        }

        $validator = Validator::make($request->all(), [
            'message'       => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()]);
        }

        $user = User::find($order->user_id);
        $general = GeneralSetting::first();

        $message                = new OrderConversation();
        $message->order_id      = $order->id;
        $message->user_id       = $user->id;
        $message->influencer_id = authInfluencerId();
        $message->sender        = 'influencer';
        $message->message       = $request->message;
        $message->save();


        $senderImagePath = getImage(getFilePath('influencerProfile') . '/' . ($influencer->image ?? ''), getFileSize('influencerProfile'));
        $sendTime = date('h:i a',strtotime($message->created_at));

        $orderData = [
            'admin_id' => 1,
            'user_id'=>$order->user_id
        ];

        foreach($orderData as $item)
        {
            $options = [
                'cluster' => $general->pusher_credential->app_cluster,
                'encrypted' => $general->pusher_credential->useTLS
            ];

            $pusher = new Pusher(
                $general->pusher_credential->app_key,
                $general->pusher_credential->app_secret,
                $general->pusher_credential->app_id,
                $options
            );

            $data = [
                'id' => $message->id,
                'message' => $message->message,
                'sender' => $message->influencer_id,
                'senderName' => $influencer->username,
                'receiver' =>  $item,
                'created_at' => $message->created_at,
                'senderImag' => $senderImagePath,
                'sendTime' => $sendTime,
            ];

            $event_name = '' . $item . '';
            $pusher->trigger($event_name, "App\\Events\\HiringChat", $data);

        }

        return response()->json($data);
    }

}
