<?php

namespace App\Http\Controllers\User;

use App\Models\Hiring;
use Pusher\Pusher;
use App\Models\Influencer;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Models\GatewayCurrency;
use App\Models\AdminNotification;
use App\Models\HiringConversation;
use App\Http\Controllers\Controller;
use App\Models\GeneralSetting;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;

class HiringController extends Controller
{
    public function hiring($name, $id)
    {
        $pageTitle  = 'Hiring Request';
        $influencer = Influencer::active()->where('id', $id)->where('username', $name)->firstOrFail();
        return view($this->activeTemplate . 'user.hiring.hiring', compact('pageTitle', 'influencer'));
    }

    public function hiringPaymentMethod(Request $request){

        $this->validation($request);

        Session::put('hiring_data', [
            'influencer_id' =>$request->influencer_id,
            'title' =>$request->title,
            'delivery_date' => $request->delivery_date,
            'amount' => $request->amount,
            'description' => $request->description,
        ]);

        $hiringData = Session::get('hiring_data');
        if (!$hiringData) {
            $notify[] = ['error', 'Session Invalidate'];
            return redirect()->route('user.home')->withNotify($notify);
        }
        $amount = $hiringData['amount'];

        $gatewayCurrency = GatewayCurrency::whereHas('method', function ($gate) {
            $gate->where('status', 1);
        })->with('method')->orderby('method_code')->get();
        $pageTitle = 'Payment Methods';

        return view($this->activeTemplate . 'user.payment.hiring_payment', compact('gatewayCurrency', 'pageTitle','amount'));

    }

    public function saveHiringData($hiringData,$paymentStatus = null){
        $user = auth()->user();
        $hiring  = new Hiring();
        $hiring->user_id  = $user->id;
        $hiring->influencer_id = $hiringData['influencer_id'];
        $hiring->title = $hiringData['title'];
        $hiring->delivery_date = $hiringData['delivery_date'];
        $hiring->amount = $hiringData['amount'];
        $hiring->description = $hiringData['description'];
        $hiring->status = 0;
        $hiring->payment_status = $paymentStatus;
        $hiring->hiring_no = getTrx();
        $hiring->save();
        return $hiring;

    }

    public function payViaWallet($hiring)
    {
        $user = auth()->user();
        $user->balance -= $hiring->amount;
        $user->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $hiring->amount;
        $transaction->post_balance = $user->balance;
        $transaction->trx_type     = '-';
        $transaction->trx          = getTrx();
        $transaction->details      = 'Deducted for hiring expense';
        $transaction->remark       = 'hiring_payment';
        $transaction->save();

        $adminNotification  = new AdminNotification();
        $adminNotification->user_id = $user->id;
        $adminNotification->title  = 'A new hiring requested by ' . $user->username;
        $adminNotification->click_url = urlPath('admin.hiring.detail', $hiring->id);
        $adminNotification->save();

        notify($user, 'HIRING_PENDING', [
            'username'      => $user->username,
            'title'         => $hiring->title,
            'amount'        => showAmount($hiring->amount),
            'hiring_no'     => $hiring->hiring_no,
        ]);

    }

    public function index(){
        $this->pageTitle = 'All Hiring';
        return $this->filterHiring();
    }

    public function pending() {
        $this->pageTitle = 'Pending Hiring';
        return $this->filterHiring('pending');
    }

    public function inprogress() {
        $this->pageTitle = 'Processing Hiring';
        return $this->filterHiring('inprogress');
    }

    public function jobDone() {
        $this->pageTitle = 'Job Done Hiring';
        return $this->filterHiring('JobDone');
    }

    public function completed() {
        $this->pageTitle = 'Completed Hiring';
        return $this->filterHiring('completed');
    }

    public function reported() {
        $this->pageTitle = 'Reported Hiring';
        return $this->filterHiring('reported');
    }

    public function cancelled() {
        $this->pageTitle = 'Cancelled Hiring';
        return $this->filterHiring('cancelled');
    }

    protected function filterHiring($scope = null) {

        $user = auth()->user();
        $hirings      = Hiring::query();

        if ($scope) {
            $hirings = $hirings->$scope();
        }
        $pageTitle = $this->pageTitle;

        $hirings = $hirings->where('user_id', $user->id)->with('influencer')->latest()->paginate(getPaginate());
        return view($this->activeTemplate . 'user.hiring.index', compact('pageTitle', 'hirings'));
    }

    protected function validation($request)
    {
        $request->validate([
            'title'         => 'required|string|max:255',
            'delivery_date' => 'required|date_format:Y-m-d|after:yesterday',
            'amount'        => 'required|numeric|gt:0',
            'description'   => 'required|string',
        ]);
    }

    public function detail($id)
    {
        $pageTitle = 'Hiring Detail';
        $user = auth()->user();
        $hiring    = Hiring::where('user_id',$user->id)->with('influencer')->findOrFail($id);
        return view($this->activeTemplate . 'user.hiring.detail', compact('pageTitle', 'hiring'));
    }

    public function completeStatus($id)
    {
        $user           = auth()->user();
        $hiring         = Hiring::JobDone()->where('id', $id)->where('user_id', $user->id)->with('influencer')->firstOrFail();
        $hiring->status = 1;
        $hiring->save();

        $influencer = $hiring->influencer;


        $influencer->balance += $hiring->amount;
        $influencer->increment('completed_order');
        $influencer->save();

        $transaction                = new Transaction();
        $transaction->influencer_id = $influencer->id;
        $transaction->amount        = $hiring->amount;
        $transaction->post_balance  = $influencer->balance;
        $transaction->trx_type      = '+';
        $transaction->details       = 'Payment received for completing a new hiring task';
        $transaction->trx           = getTrx();
        $transaction->remark        = 'hiring_payment';
        $transaction->save();

        notify($influencer, 'HIRING_COMPLETED_INFLUENCER', [
            'amount'        => showAmount($hiring->amount),
            'hiring_no'     => $hiring->hiring_no,
            'title'         => $hiring->title,
        ]);

        $notify[] = ['success', 'Hiring completed successfully'];
        return back()->withNotify($notify);
    }

    public function reportStatus(Request $request, $id)
    {
        $request->validate([
            'reason' => 'required|string',
        ]);

        $user           = auth()->user();
        $hiring         = Hiring::JobDone()->where('id', $id)->where('user_id', $user->id)->with('influencer')->firstOrFail();
        $hiring->status = 4;
        $hiring->reason = $request->reason;
        $hiring->save();

        $influencer = $hiring->influencer;

        notify($influencer, 'HIRING_REPORTED', [
            'username'      => $user->username,
            'amount'        => showAmount($hiring->amount),
            'hiring_no'     => $hiring->hiring_no,
            'title'         => $hiring->title,
            'reason'        => $hiring->reason,
        ]);

        $adminNotification  = new AdminNotification();
        $adminNotification->user_id  = $user->id;
        $adminNotification->title = 'This hiring is reported by ' . $user->username;
        $adminNotification->click_url = urlPath('admin.hiring.detail', $hiring->id);
        $adminNotification->save();

        $notify[] = ['success', 'Hiring request has been reported.'];
        return back()->withNotify($notify);
    }

    public function conversation($id)
    {
        $pageTitle           = 'Conversation';
        $hiring              = Hiring::where('user_id', auth()->id())->with('hiringMessage')->findOrFail($id);
        $influencer          = Influencer::where('id', $hiring->influencer_id)->first();
        $conversationMessage = HiringConversation::with('influencer', 'admin', 'user')->where('hiring_id', $hiring->id)->latest()->get();
        return view($this->activeTemplate . 'user.hiring.conversation', compact('pageTitle', 'conversationMessage', 'influencer', 'hiring'));
    }

    public function conversationStore(Request $request)
    {
        $user = auth()->user();
        $hiring = Hiring::where('user_id', $user->id)->find($request->hiring_id);

        if (!$hiring) {
            return response()->json(['error' => 'Hiring id not found.']);
        }

        $validator = Validator::make($request->all(), [
            'message'       => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()]);
        }

        $influencer = Influencer::active()->where('id', $hiring->influencer_id)->first();

        if (!$influencer) {
            return response()->json(['error' => 'Influencer is banned from admin.']);
        }

        $general = GeneralSetting::first();

        $message                = new HiringConversation();
        $message->hiring_id     = $hiring->id;
        $message->user_id       = $user->id;
        $message->influencer_id = $influencer->id;
        $message->sender        = 'client';
        $message->message       = $request->message;
        $message->save();

        $senderImagePath = getImage(getFilePath('userProfile') . '/' . ($user->image ?? ''), getFileSize('userProfile'));
        $sendTime = date('h:i a',strtotime($message->created_at));

        $orderData = [
            'admin_id' => 1,
            'influencer_id'=>$hiring->influencer_id
        ];

        foreach($orderData as $item)
        {
            $options = [
                'cluster' => $general->pusher_credential->app_cluster,
                'encrypted' => $general->pusher_credential->useTLS
            ];

            $pusher = new Pusher(
                $general->pusher_credential->app_key,
                $general->pusher_credential->app_secret,
                $general->pusher_credential->app_id,
                $options
            );

            $data = [
                'id' => $message->id,
                'message' => $message->message,
                'sender' => $message->user_id,
                'senderName' => $user->username,
                'receiver' =>  $item,
                'created_at' => $message->created_at,
                'senderImag' => $senderImagePath,
                'sendTime' => $sendTime,
            ];

            $event_name = '' .$item . '';
            $pusher->trigger($event_name, "App\\Events\\HiringChat", $data);
        }

        return response()->json($data);
    }


}
